/*
 * Decompiled with CFR 0.152.
 */
package edu.ku.brc.util;

import edu.ku.brc.af.core.UsageTracker;
import edu.ku.brc.exceptions.ExceptionTracker;
import edu.ku.brc.util.DataCacheIFace;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Properties;
import java.util.UUID;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpException;
import org.apache.commons.httpclient.HttpMethod;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.log4j.Logger;

public class FileCache
implements DataCacheIFace {
    private static final Logger log = Logger.getLogger(FileCache.class);
    private static String mappingFileComment = "edu.ku.brc.util.FileCache Name Mapping File";
    private static String accessTimeFileComment = "edu.ku.brc.util.FileCache Access Times File";
    private static String defaultPrefix = "brc-";
    private static String defaultSuffix = ".cache";
    private static String defaultPath = System.getProperty("java.io.tmpdir");
    protected File cacheDir;
    protected String mappingFilename;
    protected String prefix;
    protected String suffix;
    protected boolean isUsingExtensions = false;
    protected Properties handleToFilenameHash;
    protected Properties handleToAccessTimeHash;
    protected int maxCacheKb;
    protected boolean enforceMaxSize;
    protected long totalCacheSize;

    public FileCache() throws IOException {
        this(defaultPath, null);
    }

    public FileCache(String mappingFilename) throws IOException {
        this(defaultPath, mappingFilename);
    }

    public FileCache(String dir, String mappingFilename) throws IOException {
        this.mappingFilename = mappingFilename;
        this.init(dir);
    }

    protected void init(String dir) throws IOException {
        this.cacheDir = new File(dir);
        if (!this.cacheDir.exists()) {
            FileUtils.forceMkdir((File)this.cacheDir);
        }
        this.handleToFilenameHash = new Properties();
        this.handleToAccessTimeHash = new Properties();
        if (this.mappingFilename != null) {
            this.loadCacheMappingFile();
            this.loadCacheAccessTimesFile();
            this.calculateTotalCacheSize();
        }
        this.prefix = defaultPrefix;
        this.suffix = defaultSuffix;
        this.enforceMaxSize = false;
        this.maxCacheKb = Integer.MAX_VALUE;
    }

    @Override
    public void clear() {
        while (this.purgeLruCacheFile()) {
        }
    }

    public synchronized void clearItem(String key) {
        String filename = (String)this.handleToFilenameHash.get(key);
        this.removeCacheItem(filename);
    }

    public String getPrefix() {
        return this.prefix;
    }

    public void setPrefix(String prefix) {
        this.prefix = prefix;
    }

    public String getSuffix() {
        return this.suffix;
    }

    public void setSuffix(String suffix) {
        this.suffix = suffix;
    }

    public boolean isUsingExtensions() {
        return this.isUsingExtensions;
    }

    public void setUsingExtensions(boolean isUsingExtensions) {
        this.isUsingExtensions = isUsingExtensions;
    }

    public int getMaxCacheSize() {
        return this.maxCacheKb;
    }

    public void setMaxCacheSize(int kilobytes) {
        this.maxCacheKb = kilobytes;
        this.enforceMaxSize = true;
    }

    public void setEnforceMaxCacheSize(boolean value) {
        this.enforceMaxSize = value;
    }

    public boolean getEnforceMaxCacheSize() {
        return this.enforceMaxSize;
    }

    public static void setDefaultPath(String defaultPath) {
        FileCache.defaultPath = defaultPath;
    }

    protected synchronized void loadCacheMappingFile() {
        File mappingFile = new File(this.cacheDir, this.mappingFilename);
        if (!this.cacheDir.exists()) {
            this.cacheDir.mkdirs();
        }
        if (mappingFile.exists()) {
            try {
                FileInputStream fis = new FileInputStream(mappingFile);
                this.handleToFilenameHash.loadFromXML(fis);
                fis.close();
            }
            catch (IOException e) {
                log.warn((Object)"Exception while loading old cache mapping data from disk.  Starting with empty cache.", (Throwable)e);
            }
        } else {
            log.warn((Object)"Unable to locate old cache mapping file.  Cache will start out empty.");
        }
    }

    protected synchronized void loadCacheAccessTimesFile() {
        String accessTimeFilename = this.getAccessTimeFilename();
        File accessTimesFile = new File(this.cacheDir, accessTimeFilename);
        if (accessTimesFile.exists()) {
            try {
                FileInputStream fis = new FileInputStream(accessTimesFile);
                this.handleToAccessTimeHash.loadFromXML(fis);
                fis.close();
            }
            catch (IOException e) {
                log.warn((Object)"Exception while loading old cache access times from disk.", (Throwable)e);
            }
        }
    }

    protected String getAccessTimeFilename() {
        String accessTimeFilename = this.mappingFilename;
        int index = this.mappingFilename.lastIndexOf(".");
        if (index != -1) {
            accessTimeFilename = this.mappingFilename.substring(0, this.mappingFilename.lastIndexOf("."));
        }
        accessTimeFilename = String.valueOf(accessTimeFilename) + "-times.xml";
        return accessTimeFilename;
    }

    protected String findKeyLRU() {
        String lruKey = null;
        long lruAccessTime = Long.MAX_VALUE;
        for (Object k : this.handleToFilenameHash.keySet()) {
            String key = (String)k;
            long time = this.getLastAccessTime(key);
            if (time >= lruAccessTime) continue;
            lruKey = key;
            lruAccessTime = time;
        }
        return lruKey;
    }

    public synchronized void saveCacheMapping() throws IOException {
        if (this.mappingFilename == null) {
            throw new RuntimeException("Cache map filename must be set before calling saveCacheMapping()");
        }
        File mappingFile = new File(this.cacheDir, this.mappingFilename);
        try {
            this.handleToFilenameHash.storeToXML(new FileOutputStream(mappingFile), mappingFileComment);
        }
        catch (IOException e) {
            log.warn((Object)"Exception while saving cache mapping data to disk.  All cache data will be lost.", (Throwable)e);
            throw e;
        }
        this.saveCacheAccessTimes();
    }

    protected synchronized void saveCacheAccessTimes() throws IOException {
        String accessTimeFilename = this.getAccessTimeFilename();
        File accessTimesFile = new File(this.cacheDir, accessTimeFilename);
        try {
            this.handleToAccessTimeHash.storeToXML(new FileOutputStream(accessTimesFile), accessTimeFileComment);
        }
        catch (IOException e) {
            log.warn((Object)"Exception while saving cache access times to disk.", (Throwable)e);
            throw e;
        }
    }

    protected synchronized File createCacheFile(String extension) throws IOException {
        return File.createTempFile(this.prefix, extension == null ? this.suffix : extension, this.cacheDir);
    }

    protected synchronized void calculateTotalCacheSize() {
        this.totalCacheSize = 0L;
        for (Object k : this.handleToFilenameHash.keySet()) {
            String key = (String)k;
            String filename = this.handleToFilenameHash.getProperty(key);
            if (filename == null) continue;
            File f = new File(filename);
            this.totalCacheSize += f.length();
        }
    }

    protected synchronized boolean purgeCacheFile(String key) {
        if (key == null) {
            return false;
        }
        String filename = this.handleToFilenameHash.getProperty(key);
        if (filename == null) {
            return false;
        }
        File f = new File(filename);
        long filesize = f.length();
        if (!f.delete()) {
            log.warn((Object)("Failed to delete cache file: " + f.getAbsolutePath()));
        }
        this.handleToFilenameHash.remove(key);
        this.handleToAccessTimeHash.remove(key);
        this.totalCacheSize -= filesize;
        return true;
    }

    protected synchronized boolean purgeLruCacheFile() {
        String oldKey = this.findKeyLRU();
        return this.purgeCacheFile(oldKey);
    }

    protected synchronized void cacheNewItem(String key, File item) {
        long currentTime = System.currentTimeMillis();
        this.handleToAccessTimeHash.setProperty(key, Long.toString(currentTime));
        Object oldValue = this.handleToFilenameHash.setProperty(key, item.getAbsolutePath());
        if (oldValue != null) {
            this.removeCacheItem((String)oldValue);
        }
        this.totalCacheSize += item.length();
        while (this.enforceMaxSize && this.totalCacheSize > (long)(this.maxCacheKb * 1000)) {
            this.purgeLruCacheFile();
        }
    }

    protected synchronized void removeCacheItem(String filename) {
        File f = new File(filename);
        long size = f.length();
        if (!f.delete()) {
            log.warn((Object)("Failed to delete old cache file: " + f.getAbsolutePath()));
        }
        this.totalCacheSize -= size;
    }

    public String cacheData(byte[] data) throws IOException {
        String key = UUID.randomUUID().toString();
        this.cacheData(key, data);
        return key;
    }

    public void cacheData(String key, byte[] data) throws IOException {
        File f = this.createCacheFile(null);
        FileOutputStream fos = new FileOutputStream(f);
        fos.write(data);
        fos.flush();
        fos.close();
        this.cacheNewItem(key, f);
    }

    public String cacheFile(File f) throws IOException {
        this.cacheFile(f.getName(), f);
        return f.getName();
    }

    public void cacheFile(String key, File f) throws IOException {
        String extension = this.isUsingExtensions ? "." + FilenameUtils.getExtension((String)f.getName()) : null;
        File cachedFile = this.createCacheFile(extension);
        log.debug((Object)String.format("Caching Key[%s]  file[%s] -> [%s]", key, f.getName(), cachedFile.getName()));
        FileUtils.copyFile((File)f, (File)cachedFile);
        this.cacheNewItem(key, cachedFile);
    }

    public String cacheWebResource(String url) throws HttpException, IOException {
        HttpClient httpClient = new HttpClient();
        GetMethod get = new GetMethod(url);
        get.setFollowRedirects(true);
        int result = httpClient.executeMethod((HttpMethod)get);
        if (result != 200) {
            log.debug((Object)("Retrieving " + url + " resulted in unexpected code: " + result));
            throw new HttpException("Unexpected HTTP code received: " + result);
        }
        byte[] response = get.getResponseBody();
        if (response.length > 0) {
            this.cacheData(url, response);
            return url;
        }
        throw new HttpException("Web request returned zero length response");
    }

    public void refreshCachedWebResource(String key) throws HttpException, IOException {
        this.cacheWebResource(key);
    }

    public File getCacheFile(String key) {
        log.debug((Object)String.format("Get [%s]", key));
        String filename = this.handleToFilenameHash.getProperty(key);
        if (filename == null) {
            return null;
        }
        File f = new File(filename);
        if (f.exists()) {
            this.handleToAccessTimeHash.setProperty(key, Long.toString(System.currentTimeMillis()));
            return f;
        }
        log.debug((Object)("Previously cached file '" + filename + "' is missing.  Cleaning up cache map data."));
        this.handleToFilenameHash.remove(key);
        return null;
    }

    public long getLastAccessTime(String key) {
        long accessTimeMillis;
        String accessTimeString = this.handleToAccessTimeHash.getProperty(key);
        if (accessTimeString == null) {
            return Long.MIN_VALUE;
        }
        try {
            accessTimeMillis = Long.parseLong(accessTimeString);
        }
        catch (NumberFormatException nfe) {
            UsageTracker.incrHandledUsageCount();
            ExceptionTracker.getInstance().capture(FileCache.class, nfe);
            log.error((Object)("Unable to parse access time for cache item: " + key), (Throwable)nfe);
            accessTimeMillis = Long.MIN_VALUE;
        }
        return accessTimeMillis;
    }

    public long getCurrentCacheSize() {
        return this.totalCacheSize;
    }

    @Override
    public void shutdown() throws Exception {
        this.saveCacheMapping();
    }
}

